/*
 *
 * Copyright (c) 2004-2012 by Rodney Kinney, Brent Easton
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License (LGPL) as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, copies are available
 * at http://www.opensource.org.
 */
package VASSAL.build.module.map.boardPicker.board;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.w3c.dom.Element;

import VASSAL.build.AbstractConfigurable;
import VASSAL.build.Buildable;
import VASSAL.build.module.Map;
import VASSAL.build.module.documentation.HelpFile;
import VASSAL.build.module.map.boardPicker.Board;
import VASSAL.build.module.map.boardPicker.board.mapgrid.GridContainer;
import VASSAL.build.module.map.boardPicker.board.mapgrid.GridNumbering;
import VASSAL.build.module.map.boardPicker.board.mapgrid.Zone;
import VASSAL.build.module.map.boardPicker.board.mapgrid.ZoneHighlight;
import VASSAL.build.module.map.boardPicker.board.mapgrid.ZonedGridHighlighter;
import VASSAL.configure.Configurer;
import VASSAL.i18n.Resources;

/**
 * Map Grid that contains any number of {@link VASSAL.build.module.map.boardPicker.board.mapgrid.Zone}s against a background {@link MapGrid}
 */
public class ZonedGrid extends AbstractConfigurable implements GeometricGrid, GridContainer {
  protected List<Zone> zones = new ArrayList<>();
  protected MapGrid background;
  protected GridContainer container;
  protected ZonedGridHighlighter zoneHighlighters;

  @Override
  public String[] getAttributeDescriptions() {
    return new String[0];
  }

  @Override
  public Class<?>[] getAttributeTypes() {
    return new Class<?>[0];
  }

  @Override
  public String[] getAttributeNames() {
    return new String[0];
  }

  @Override
  public String getAttributeValueString(String key) {
    return null;
  }

  @Override
  public void setAttribute(String key, Object value) {
  }

  @Override
  public Configurer getConfigurer() {
    return null;
  }

  @Override
  public void addTo(Buildable parent) {
    container = (GridContainer) parent;
    container.setGrid(this);
  }

  public GridContainer getContainer() {
    return container;
  }

  @Override
  public Dimension getSize() {
    return container.getSize();
  }

  @Override
  public boolean contains(Point p) {
    return container.contains(p);
  }

  @Override
  public void removeGrid(MapGrid grid) {
    if (background == grid) {
      background = null;
    }
  }

  @Override
  public Board getBoard() {
    return container != null ? container.getBoard() : null;
  }

  public Map getMap() {
    return getBoard() == null ? null : getBoard().getMap();
  }

  @Override
  public void setGrid(MapGrid grid) {
    background = grid;
  }

  @Override
  public Class<?>[] getAllowableConfigureComponents() {
    return background == null ?
      new Class<?>[]{
        Zone.class,
        HexGrid.class,
        SquareGrid.class,
        RegionGrid.class
      } :
      new Class<?>[]{Zone.class};
  }

  public static String getConfigureTypeName() {
    return Resources.getString("Editor.MultiZoneGrid.component_type"); //$NON-NLS-1$
  }

  @Override
  public HelpFile getHelpFile() {
    return HelpFile.getReferenceManualPage("ZonedGrid.html"); //$NON-NLS-1$
  }

  @Override
  public void removeFrom(Buildable parent) {
    ((GridContainer) parent).removeGrid(this);
  }

  /*
   * Zones that do not use the background grid must clip out the portion of
   * the background grid that they cover. Cache as much of the work as
   * possible to prevent bogging down with large numbers of zones.
   */
  protected Area scaledZones = null;
  protected Area translatedZones = null;
  protected AffineTransform scaleTransform;
  protected AffineTransform translateTransform;
  protected double lastScale = 0.0;
  protected int lastX = -1;
  protected int lastY = -1;

  @Override
  public void draw(Graphics g, Rectangle bounds, Rectangle visibleRect, double scale, boolean reversed) {
    /*
     * Skip clipping if there is no background grid, or it isn't visible
     */
    if (background != null && background.isVisible()) {
      /*
       * Calculate and cache scaled shape consisting of all zones that do not
       * use the parent grid. (There may be none!)
       */
      if (lastScale != scale || scaleTransform == null) {
        scaleTransform = AffineTransform.getScaleInstance(scale, scale);
        scaledZones = null;

        for (final Zone zone : zones) {
          if (!zone.isUseParentGrid()) {
            if (scaledZones == null) {
              scaledZones = new Area(
                scaleTransform.createTransformedShape(zone.getShape()));
            }
            else {
              scaledZones.add(new Area(
                scaleTransform.createTransformedShape(zone.getShape())));
            }
          }
        }
        lastScale = scale;
        translateTransform = null;  // Force translatedZones to be regenerated
      }

      /*
       * Translate and cache the combined zone shape
       */
      if (lastX != bounds.x || lastY != bounds.y || translateTransform == null) {
        translateTransform = AffineTransform.getTranslateInstance(bounds.x, bounds.y);
        translatedZones = null;
        if (scaledZones != null) {
          translatedZones = new Area(translateTransform.createTransformedShape(scaledZones));
        }
        lastX = bounds.x;
        lastY = bounds.y;
      }

      /*
       * Clip out the area covered by the Zones not using the background grid and draw it.
       */
      final Graphics2D g2d = (Graphics2D) g;
      final Shape oldClip = g2d.getClip();
      if (translatedZones != null && oldClip != null) {
        final Area clipArea = new Area(oldClip);
        clipArea.subtract(translatedZones);
        g2d.setClip(clipArea);
      }
      background.draw(g, bounds, visibleRect, scale, reversed);
      g2d.setClip(oldClip);
    }

    /*
     * Draw each Zone
     */
    for (final Zone zone : zones) {
      zone.draw(g, bounds, visibleRect, scale, reversed);
    }
  }

  @Override
  public GridNumbering getGridNumbering() {
    return background != null ? background.getGridNumbering() : null;
  }

  @Override
  public Point getLocation(String location) throws BadCoords {
    for (final Zone zone : zones) {
      try {
        final Point p = zone.getLocation(location);
        if (p != null && zone.contains(p)) {
          return p;
        }
      }
      catch (final BadCoords bc) {
      }
    }
    if (background != null)
      return background.getLocation(location);
    else
      throw new BadCoords();
  }

  @Override
  public boolean isVisible() {
    return true;
  }

  @Override
  public String locationName(Point p) {
    String name = null;
    for (final Zone zone : zones) {
      if (zone.contains(p)) {
        name = zone.locationName(p);
        break;
      }
    }
    if (name == null
        && background != null) {
      name = background.locationName(p);
    }
    return name;
  }

  @Override
  public String localizedLocationName(Point p) {
    String name = null;
    for (final Zone zone : zones) {
      if (zone.contains(p)) {
        name = zone.localizedLocationName(p);
        break;
      }
    }
    if (name == null
        && background != null) {
      name = background.localizedLocationName(p);
    }
    return name;
  }

  @Override
  public int range(Point p1, Point p2) {
    MapGrid grid = background;
    final Zone z1 = findZone(p1);
    final Zone z2 = findZone(p2);
    if (z1 == z2
      && z1 != null
      && z1.getGrid() != null) {
      grid = z1.getGrid();
    }
    return grid != null ? grid.range(p1, p2) : (int)Math.round(p1.distance(p2));
  }

  @Override
  public Area getGridShape(Point center, int range) {
    Area a = null;
    final Zone z = findZone(center);
    if (z != null
      && z.getGrid() instanceof GeometricGrid) {
      a = ((GeometricGrid)z.getGrid()).getGridShape(center, range);
    }
    if (a == null
      && background instanceof GeometricGrid) {
      a = ((GeometricGrid)background).getGridShape(center, range);
    }
    if (a == null) {
      a = new Area(new Ellipse2D.Double(center.x - range, center.y - range, range * 2, range * 2));
    }
    return a;
  }

  public Zone findZone(Point p) {
    for (final Zone zone : zones) {
      if (zone.contains(p)) {
        return zone;
      }
    }
    return null;
  }

  public Zone findZone(String name) {
    for (final Zone zone : zones) {
      if (zone.getName().equals(name)) {
        return zone;
      }
    }
    return null;
  }

  @Override
  public Point snapTo(Point p) {
    Point snap = null;
    final Zone z = findZone(p);
    if (z != null) {
      snap = z.snapTo(p);
    }
    if (snap == null) {
      snap = background != null ? background.snapTo(p) : p;
    }
    return snap;
  }

  @Override
  public boolean isLocationRestricted(Point p) {
    for (final Zone zone : zones) {
      if (zone.contains(p)) {
        return zone.getGrid() != null && zone.getGrid().isLocationRestricted(p);
      }
    }
    return background != null && background.isLocationRestricted(p);
  }

  public void addZone(Zone z) {
    zones.add(z);
  }

  public void removeZone(Zone z) {
    zones.remove(z);
  }

  public Iterator<Zone> getZones() {
    return zones.iterator();
  }

  public MapGrid getBackgroundGrid() {
    return background;
  }

  public void setBackgroundGrid(MapGrid background) {
    this.background = background;
  }

  @Override
  public void build(Element e) {
    super.build(e);
    if (getComponentsOf(ZonedGridHighlighter.class).isEmpty()) {
      addChild(new ZonedGridHighlighter());
    }
  }

  private void addChild(Buildable b) {
    add(b);
    b.addTo(this);
  }

  public void setZoneHighlighter(ZonedGridHighlighter zh) {
    zoneHighlighters = zh;
  }

  public ZoneHighlight getZoneHighlight(String name) {
    if (zoneHighlighters != null) {
      return zoneHighlighters.getZoneHighlightByName(name);
    }
    return null;
  }
}
